function [f,Yhat,a,Psi,Q_sq,V,S] = pce_poly(X,Y,pmax,id)
%{
--------------------------------------------------------------------------
Compute coefficients of full d_x-dimensional PCE of order p based on
OLS of (X,Y), where X ~ Gauss(0,1)
--------------------------------------------------------------------------
Created by:
Fong-Lin Wu (fonglin.wu@tum.de)
Max Ehre (max.ehre@tum.de)
Technische Universitat Munchen
www.era.bgu.tum.de
--------------------------------------------------------------------------
Version: 
* 2019-08 Transition from Matlab
--------------------------------------------------------------------------
Input:
* X    : Design input generated by Latin-Hypercube
* Y    : Result from Finite-Element-Method
* pmax : Maximum polynomial order of PCE-PLS 
--------------------------------------------------------------------------
Output:
* f    : Lamda function of PCE-PLS model
* Yhat : Result output by PCE-PLS model
* a    : Coefficient of Hermite polynomials
* Psi  : Products of Hermite polynomials
* Q_sq : 1 - (Leave-One-Out error)
--------------------------------------------------------------------------
Based on:

--------------------------------------------------------------------------
%}

% read DoE dimensions
[~ , d_x] = size(X);
[n_s,~] = size(Y);

%% Hermite polynomials by recursive formula

He_p    = cell(pmax,1);
He_p{1} = 1;       % H_1 = 1
He_p{2} = [1 0];   % H_2 = x
for n = 2:pmax+1
   He_p{n+1} = [He_p{n} 0] - (n-1)*[0 0 He_p{n-1}];   % recursive formula
end

He_p_mat = zeros(pmax+1);

for row=1:pmax+1
    He_p_mat(row,:) = [zeros(1,pmax+1-row),He_p{row}];
end

% multi-index
alpha  = multi_index(d_x,pmax);
P      = size(alpha,1);

for i=1:d_x
    Psi_v(:,:,i) = cell2mat(arrayfun(@(index) polyval(He_p_mat(alpha(index,i)+1,:),X(:,i)),1:P,'un',0));
end

norm = sqrt(prod(factorial(alpha),2));
Psi  = prod(Psi_v,3)./norm';

% information matrix
M  = Psi'*Psi;

% inverse information matrix
MI = inv(M);

% projection matrix
PP  = Psi*MI*Psi';

% annihilator matrix
AA  = eye(n_s) - PP;

% LOO-error approximation
h       = diag(PP);
eps_LOO = mean( ( (AA*Y)./(1-h) ).^2 ) / var(Y) * (1 + trace(MI))*n_s/(n_s-P);
Q_sq    = 1 - eps_LOO;
    
%% build predictive model

a         = (Psi'*Psi) \ (Psi'*Y) ;

function output = pce_constructor(input)
    output = 0;
    for i = 1:size(alpha,1)
        Psi_i = 1;
        for j = 1:size(alpha,2)
            Psi_i = Psi_i.*polyval(He_p_mat(alpha(i,j)+1,:),input(:,j));
        end
        output = output + a(i)*Psi_i/norm(i);
    end
    
end

f = @pce_constructor;

% f = @(X) prod(reshape(cell2mat(arrayfun(@(dim) cell2mat(arrayfun(@(pol) ...
%          polyval(He_p_mat(alpha(pol,dim)+1,:),X(:,dim)),1:size(alpha,1),'un',0)),1:d_x,'un',0))...
%          ,size(X,1),size(alpha,1),d_x),3)./sqrt(prod(factorial(alpha),2))'*a;
%          
Yhat    = f(X);

if exist('id','var')
    [V,S] = pce_sensitivities(a,alpha,id);
else
    V = []; S = [];
end

end