 %{
---------------------------------------------------------------------------
PCE-driven PLS algorithm - Linear version
---------------------------------------------------------------------------
Created by:
Fong-Lin Wu (fonglin.wu@tum.de)
Max Ehre (max.ehre@tum.de)
Technische Universitat Munchen
www.era.bgu.tum.de
---------------------------------------------------------------------------
Version: 
* 2019-08 Transition from Matlab
---------------------------------------------------------------------------
Input:
* X      : Design input generated by Latin-Hypercube
* Y      : Result from Finite-Element-Method
* d      : Maximum number of components (directions)
* pmax   : Maximum polynomial order of PCE-PLS
---------------------------------------------------------------------------
Output:
* f      : Lamda function of PCE-PLS model
* Yhat   : Result outpur by PCE-PLS model
* orthogonal_error: Orthogonal error of R matrix
* d_red  : Number of used PCE directions 
* pfinal : Final order of polynomial chaos expansion
---------------------------------------------------------------------------
Based on: 
* PLS-based adaptation for efficient PCE representation in high dimensions.
Iason Papaioannou , Max Ehre, Daniel Straub
Engineering Risk Analysis Group, Technische Universita?t Mu?nchen, Arcisstr. 21, 80290 Mu?nchen, Germany
---------------------------------------------------------------------------
%}

function [f,Yhat,orthogonal_error,d_red,pfinal] = pls_pce_R_Lin(X,Y,d,pmax)

% PLS-driven PCE with polyval-based 1-D PCE evaluation
n = size(X,1);
m = size(X,2);

meanX = mean(X,1);
meanY = mean(Y,1);
X0 = bsxfun(@minus, X, meanX);
Y0 = bsxfun(@minus, Y, meanY);

Wload = zeros(m,d);
Pload = zeros(m,d);
Yvar = zeros(d,1);
Xscore = zeros(n,d);

E = X0;
F = Y0;

for i=1:d

    % compute normalized weight
    Wload0 = E'*F;
    alpha = norm(Wload0);
    Wload0 = Wload0/alpha;
    
    % compute score
    Xscore0 = E*Wload0;

    % we assume linear responses
    Wload(:,i) = Wload0;         
    Xscore(:,i) = Xscore0;
    pfinal(i) = 1;
    
    % fit a 1-D PCE
    [g_pce1{i}, Freg, a ,~] = pce_poly(Xscore(:,i),F,pfinal(i));
         
    aout{i} = a;    
    Yvar(i) = var(Freg)/var(Y0);  
    Pload(:,i) = Xscore(:,i)'*E / (Xscore(:,i)'*Xscore(:,i));
    
    if Yvar(i)/sum(Yvar) < 0.005
        break;
    end
    
    E = E - Xscore(:,i) * Pload(:,i)';
    F = F - Freg;
    
end

d_red = i;

Wload = Wload(:,1:d_red);
Pload = Pload(:,1:d_red);
Xscore = Xscore(:,1:d_red);
Yvar = Yvar(1:d_red);

R = Wload / (Pload'*Wload);

orthogonal_error = eye(i)-R'*R;

f    = @(X) meanY + sum(cell2mat(arrayfun(@(index) g_pce1{index}((X - meanX)*R(:,index)),1:i,'un',0)),2);
Yhat = f(X);

return