 %{
---------------------------------------------------------------------------
PCE-driven PLS algorithm - Non Linear version
---------------------------------------------------------------------------
Created by:
Fong-Lin Wu (fonglin.wu@tum.de)
Max Ehre (max.ehre@tum.de)
Technische Universitat Munchen
www.era.bgu.tum.de
---------------------------------------------------------------------------
Version: 
* 2019-08 Transition from Matlab
---------------------------------------------------------------------------
Input:
* X      : Design input generated by Latin-Hypercube
* Y      : Result from Finite-Element-Method
* d      : Maximum number of components (directions)
* pmax   : Maximum polynomial order of PCE-PLS
---------------------------------------------------------------------------
Output:
* f      : Lamda function of PCE-PLS model
* Yhat   : Result outpur by PCE-PLS model
* orthogonal_error: Orthogonal error of R matrix
* d_red  : Number of used PCE directions 
* pfinal : Final order of polynomial chaos expansion
---------------------------------------------------------------------------
Based on: 
* PLS-based adaptation for efficient PCE representation in high dimensions.
Iason Papaioannou , Max Ehre, Daniel Straub
Engineering Risk Analysis Group, Technische Universita?t Mu?nchen, Arcisstr. 21, 80290 Mu?nchen, Germany
---------------------------------------------------------------------------
%}

function [f,Yhat,orthogonal_error,d_red,pfinal] = pls_pce_R_NLin(X,Y,d,pmax)

% PLS-driven PCE with polyval-based 1-D PCE evaluation
n = size(X,1);
m = size(X,2);

tol = 1.e-3;
maxiter = 100;

% center matrices
meanX = mean(X,1);
meanY = mean(Y,1);
X0 = bsxfun(@minus, X, meanX);
Y0 = bsxfun(@minus, Y, meanY);

Wload = zeros(m,d);
Pload = zeros(m,d);
Yvar = zeros(d,1);
Xscore = zeros(n,d);

E = X0;
F = Y0;

for i=1:d

    % compute normalized weight
    Wload0 = E'*F;
    alpha = norm(Wload0);
    Wload0 = Wload0/alpha;
    
    % compute score
    Xscore0 = E*Wload0;
 
    % try with different order PCE
    for p = 1:pmax
        
        Wloadp(:,p) = Wload0;
        Xscorep0 = Xscore0;

        % we assume non-linear response
        for iter=1:maxiter
            
            [g_tmp,Freg,a{p},Psi] = pce_poly(Xscorep0,F,p);
            
            err = F-Freg;
            
            Z = E.*repmat(Psi(:,1:p)*(a{p}(2:p+1).*sqrt([1:p]')),[1 m]);
            
            dW = pinv(Z'*Z)* Z' *err;
            
            Wloadp(:,p) = Wloadp(:,p) + dW;
            Wloadp(:,p) = Wloadp(:,p)/norm(Wloadp(:,p));
            Xscorep1 = E*Wloadp(:,p);
            
            dXscorep = norm(Xscorep1-Xscorep0)/norm(Xscorep0);
            
            if dXscorep < tol
                Xscorep(:,p) = Xscorep1;
                break;
            else
                Xscorep0 = Xscorep1;
            end
            
        end
        
        if iter == maxiter
            % maximum iterations reached: use pls direction
            Wloadp(:,p) = Wload0;
            Xscorep(:,p) = Xscore0;
        end
        
        [g_pcep{p}, Fregp(:,p), a{p}, ~, Q2(p)] = pce_poly(Xscorep(:,p),F,p);
        
        if Q2(p) > 1-1.e-4
            break; 
        end
      
    end
    
    % choose the one with lowest leave-one-out error
    pfinal(i) = find(Q2 == max(Q2(1:p)));
    
    % assign final result to current the direction 
    Wload(:,i) = Wloadp(:,pfinal(i));
    Xscore(:,i) = Xscorep(:,pfinal(i));
    Freg = Fregp(:,pfinal(i));
    g_pce1{i} = g_pcep{pfinal(i)};
    aout{i}   = a{pfinal(i)};

    Yvar(i) = var(Freg)/var(Y0);
    Pload(:,i) = Xscore(:,i)'*E / (Xscore(:,i)'*Xscore(:,i));
    
    if Yvar(i)/sum(Yvar) < 0.005
        break;
    end
    
    E = E - Xscore(:,i) * Pload(:,i)';
    F = F - Freg;
    
end

d_red = i;

Wload = Wload(:,1:d_red);
Pload = Pload(:,1:d_red);
Xscore = Xscore(:,1:d_red);
Yvar = Yvar(1:d_red);

R = Wload / (Pload'*Wload);

orthogonal_error = eye(i)-R'*R;

f    = @(X) meanY + sum(cell2mat(arrayfun(@(index) g_pce1{index}((X - meanX)*R(:,index)),1:i,'un',0)),2);
Yhat = f(X);

return